﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web.UI;
using System.Web.UI.WebControls;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager;
using VeteransAffairs.Registries.BusinessManager.TBI;
using BusinessHelper = VeteransAffairs.Registries.Business.Utilities.Helpers;

public partial class TBI_Instruments_PatientGlobalImpOfChange : BaseTBIInstrumentPage {

    #region Variable Declarations

    PatientManager patientManager = new PatientManager();
    InstrumentType instrumentType = new InstrumentType();

    protected int _surveyTypeId = 10; // Database Id for Patients Global Impression of Change (PGIC) scale 

    SURVEY _survey;   // the root object in the Survey set
    SURVEY_RESULT _surveyResult; // the object representing the actual data in the Survey

    List<STD_QUESTION> QuestionList = new List<STD_QUESTION>();

    #endregion

    /// <summary>
    /// Default Constructor
    /// </summary>
    public TBI_Instruments_PatientGlobalImpOfChange() {
        //Manager Save Notification Event Handler
        patientManager.BOSaveSuccess += manager_BOSaveSuccess;
    }

    #region Protected Methods

    /// <summary>
    /// Page Load method
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void Page_Load(object sender, EventArgs e) {
        TBIServiceTO _tbiServiceTo; //comment

        try {
            // confirm instrument is active - if not, a redirect will occur.
            this.CheckActiveStatus(_surveyTypeId);

            _tbiServiceTo = (TBIServiceTO)Session["TBIServiceTO"];
            _tbiInstrumentManager.TbiServiceTo = _tbiServiceTo;

            LabelBreadCrumb.Text = "TBI Instruments > Instrument Associations > "
                                  + _tbiInstrumentManager.GetInstrumentByID(_surveyTypeId).Name;

            if (_tbiInstrumentManager.TbiServiceTo != null) {
                Label labelUser = (Label)Master.FindControl("LabelCurrentUser");
                labelUser.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.UserName);

                LabelPatient.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientName);
                LabelSSN.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientSnum);
            }
        } catch (Exception ex) {
            //Redirect to message that session is ended and need to log back in from CPRS.
            Exception _tbilogerror = HandleTBIException(ex, "MedRed_L-Test.aspx", "100027", "SESSIOENDED");
            Response.Redirect("Warning.aspx?err=2", false);
        }

        if (!IsPostBack) {
            if (instrumentType.Status != null) {
                _tbiInstrumentManager.TbiServiceTo.SelectedInstrumentType = instrumentType;
            }

            string _snum = "";
            if (_tbiInstrumentManager.TbiServiceTo != null) {
                _snum = _tbiInstrumentManager.TbiServiceTo.PatientSnum;
            }

            if (!string.IsNullOrEmpty(_snum))
            {
                string patICN = patientManager.GetPatientICNBySnum(_snum);

                if (!string.IsNullOrEmpty(patICN)) {
                    ViewState["PatientICN"] = patICN;
                    RefreshSurvey(patICN);
                } else {
                    throw new ArgumentException("Patient with this SSN does not exist.");
                }

                if (_survey != null && !_survey.isNew) {
                    SetUpSurvey(_survey);
                }
                string referrer = "";
                if (Request.UrlReferrer != null) {
                    referrer = Request.UrlReferrer.PathAndQuery;
                }
                ViewState["referrer"] = referrer;
            }

            //Check if the request is to start a new item 
            //a new item page request is signified when the querystring parameter id equals zero or is not present
            if (_tbiInstrumentManager.TbiServiceTo == null)
            {
                Logging.WriteLogToFile("PatientGlobalImpOfChange.Page_Load(): WarningRedirect: _tbiInstrumentManager.TbiServiceTo == null");
                //Redirect to message that session is ended and need to log back in from CPRS.
                Response.Redirect("Warning.aspx?err=2");
            } else {
                LabelPatient.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientName);
                LabelSSN.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientSnum.Trim().Length == 9 ? String.Format("{0:000-00-0000}", int.Parse(_tbiInstrumentManager.TbiServiceTo.PatientSnum)) : _tbiInstrumentManager.TbiServiceTo.PatientSnum);
            }
        }

        string instrumentTitle = GetInstrumentTitle(_surveyTypeId);
        LabelInstrumentTitle.Text = instrumentTitle;
        Page.Title = "TBI Instruments - " + instrumentTitle;

        QuestionList = GetAllQuestionsForSurvey();

        // Set control values for Diagnosis Question
        ctrlDiagnosis.QuestionText = GetQuestionText("DX");
        ctrlDiagnosis.SurveyTypeCode = HiddenSurveyType.Value;

        // Set labels, etc
        LabelDate1.Text = GetQuestionText("D");
        LabelForQ1.Text = GetQuestionText("1");
        LabelForQ2.Text = GetQuestionText("2");
        LabelForQ3.Text = GetQuestionText("3");
        cntlRehabilitationProviderSelector.QuestionText = GetQuestionText("RP");
        cntlRehabilitationProviderSelector.SurveyTypeCode = HiddenSurveyType.Value;

        cntlTypeOfServiceSelector.QuestionText = GetQuestionText("TS");
        cntlTypeOfServiceSelector.SurveyTypeCode = HiddenSurveyType.Value;

        dpDate1.MaxDate = DateTime.Now;
    }

    protected override void LoadViewState(object savedState) {
        base.LoadViewState(savedState);
        string surveyXml = (string)ViewState["surveyXml"];
        _survey = SerializerUtil.Deserialize<SURVEY>(surveyXml);
    }

    protected override object SaveViewState() {
        ViewState["surveyXml"] = _survey.Serialize();
        return base.SaveViewState();
    }

    /// <summary>
    /// Returns the Question_Text of the question with the passed Question_Number field
    /// </summary>
    /// <param name="questionId"></param>
    /// <returns></returns>
    protected string GetQuestionText(string questionId) {
        foreach (STD_QUESTION ques in QuestionList) {
            if (ques.QUESTION_NUMBER.ToLower() == questionId.ToLower()) {
                return ques.QUESTION_TEXT;
            }
        }

        return "";
    }

    /// <summary>
    /// Returns the Question_Id of the question with the passed Question_Number field
    /// </summary>
    /// <param name="questionId"></param>
    /// <returns></returns>
    protected int GetQuestionId(string questionNumber) {
        foreach (STD_QUESTION ques in QuestionList) {
            if (ques.QUESTION_NUMBER.ToLower() == questionNumber.ToLower()) {
                return ques.ID;
            }
        }

        return 0;
    }

    /// <summary>
    /// Save Draft button clicked.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    public void btnSaveSurveyDraft_Click(object sender, EventArgs e) {
        Int32 status = 0;
        SaveFailMessage = "Save Failed. Please press " + btnSaveSurveyDraft.Text + " button again to avoid data loss.";
        SaveCompleteSurvey(ref status);
        SetUpSurvey(_survey);
    }

    /// <summary>
    /// Save and Prepare Note button clicked.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void btnSubmit_Click(object sender, EventArgs e) {
        Int32 status = 0;
        SaveFailMessage = "Save Failed. Please press " + btnSubmitSurvey.Text + " button again to avoid data loss.";
        SaveCompleteSurvey(ref status);

        if (status > 0) {
            Session["survey"] = _survey;
            //Session["rawScore"] = hiddenRawScore.Value.ToString();
            //Session["tScore"] = hiddenTScore.Value.ToString();
            Response.Redirect("InstrumentSummaryNote.aspx", false);
        } else {
            SetUpSurvey(_survey); //Follow Save Draft logic 
        }
    }

    protected void InActiveTimer_Tick(object sender, EventArgs e) {
        Int32 status = 0;
        SaveFailMessage = "Save Failed. Please press " + btnSaveSurveyDraft.Text + " button again to avoid data loss.";

        Page.Validate();

        if (Page.IsValid) {
            SaveCompleteSurvey(ref status);
            //SetUpSurvey(_survey);
            Response.Redirect("PatientGlobalImpOfChange.aspx", false);
        } else {
            SaveFailMessage = "Auto-Save Validation Failed. Please complete required fields in the survey and press " + btnSaveSurveyDraft.Text + " button again to avoid data loss.";
        }
    }

    protected void btnCancel_Click(object sender, EventArgs e) {
        ViewState["surveyXml"] = null;
        ViewState["PatientICN"] = null;
        ViewState["referrer"] = null;
        Session.Remove("survey");
        Response.Redirect("Instrument.aspx", false);
        //this.ClientScript.RegisterStartupScript(typeof(Page), "closePage", "<script type='text/javascript'> window.open('','_self','');window.close(); </script>");
    }

    #endregion

    #region Private Methods

    /// <summary>
    /// Get all questions for this survey
    /// </summary>
    /// <returns></returns>
    private List<STD_QUESTION> GetAllQuestionsForSurvey() {
        return _tbiInstrumentManager.SelectAllQuestions(_surveyTypeId).ToList();
    }

    /// <summary>
    /// Refresh Survey
    /// </summary>
    /// <param name="patientICN"></param>
    private void RefreshSurvey(string patientICN) {
        var surveys = _tbiInstrumentManager.SelectByPatientICNAndSurveyType(patientICN, _surveyTypeId, true).ToList();

        if (surveys.Count > 0) {
            _survey = surveys.First();
        } else {
            _survey = new SURVEY();
        }

        _survey.SetAsChangeTrackingRoot();
    }

    /// <summary>
    /// Save Complete Survey
    /// </summary>
    /// <param name="status"></param>
    private void SaveCompleteSurvey(ref Int32 status) 
    {
        Page.Validate();
        if (Page.IsValid) {
            string patientICN = String.Empty;

            status = 0;

            //Retrieve patient's ICN from the session variable
            if (ViewState["PatientICN"] != null) {
                patientICN = ViewState["PatientICN"].ToString();
                if (String.IsNullOrEmpty(patientICN.Trim()))
                {
                    Logging.WriteLogToFile("PatientGlobalImpOfChange.SaveCompletedSurvey(): WarningRedirect: String.IsNullOrEmpty(patientICN.Trim()");
                    //Lost state. Redirect to message that need to log back in from CPRS.
                    Response.Redirect("Warning.aspx?err=2", false);
                }
            }
            else
            {
                Logging.WriteLogToFile("PatientGlobalImpOfChange.SaveCompletedSurvey(): WarningRedirect: ViewState[PatientICN] != null");
                //Lost state. Redirect to message that need to log back in from CPRS.
                Response.Redirect("Warning.aspx?err=2");
            }

            if (_survey != null) {
                if (_survey.isNew) {
                    //Ensure that a concurrent session hasn't already inserted a record for this survey
                    var surveys = _tbiInstrumentManager.SelectByPatientICNAndSurveyType(patientICN, _surveyTypeId, true).ToList();

                    if (surveys.Count > 0)
                    {
                        Logging.WriteLogToFile("PatientGlobalImpOfChange.SaveCompletedSurvey(): WarningRedirect: surveys.Count > 0");
                        //A concurrent session has already inserted a record for this survey and it is not new anymore.
                        Response.Redirect("Warning.aspx?err=9", false);
                    } else {
                        _survey.SURVEY_STATUS = "inprocess";

                        _survey.SetAsInsertOnSubmit();

                        _survey.patient_ICN = patientICN;

                        _survey.STD_SURVEY_TYPE_ID = _surveyTypeId;

                        // TODO: change to datepicker
                        _survey.SURVEY_DATE = DateTime.Today;

                        //Save survey if it's new before adding any responses
                        SuppressSaveMessage = true;
                        _tbiInstrumentManager.Update(_survey);
                        SuppressSaveMessage = false;

                        RefreshSurvey(patientICN);
                    }
                }

                SaveSurvey(ref _survey);

                _survey.SetAsUpdateOnSubmit();
                status = _tbiInstrumentManager.Update(_survey);

                //Fortify errors on this line (this.Master as RegistryInstrumentMaster).SetDirtyFlag(false);
                ((RegistryInstrumentMaster)Master).SetDirtyFlag(false);

                RefreshSurvey(patientICN);
                SuppressSaveMessage = false;
            }
        }
    }

    /// <summary>
    /// Save survey.
    /// </summary>
    /// <param name="survey"></param>
    private void SaveSurvey(ref SURVEY survey) {
        List<STD_QUESTION> questions = GetAllQuestionsForSurvey();

        //Delete all the survey results from database if there are any.
        SURVEY surveyWithResults = _tbiInstrumentManager.GetSurveyWithAllAnswers(survey.SURVEYS_ID);
        _tbiInstrumentManager.DeleteAllSurveyResults(surveyWithResults);

        // Question DX
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question DX choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(ctrlDiagnosis.GetSelectedId());
            _surveyResult.RESULT_TEXT = ctrlDiagnosis.GetSelectedResultText();
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "PatientGlobalImpOfChange.aspx", "100027", "SaveSurvey() - Question DX");
        }

        // Question D - Date
        STD_QUESTION questionD = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "D");
        _surveyResult = new SURVEY_RESULT();

        try {
            // Add Question D choice
            _surveyResult.STD_QUESTION_CHOICE_ID = questionD.STD_QUESTION_CHOICEs.FirstOrDefault().STD_QUESTION_CHOICE_ID;
            _surveyResult.RESULT_TEXT = ((DateTime)dpDate1.SelectedDate).ToShortDateString();
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "PatientGlobalImpOfChange.aspx", "100027", "SaveSurvey() - QuestionD");
        }

        // Question 1 - Chief complaint
        if (!String.IsNullOrEmpty(txtDiagnosis.Text)) {
            STD_QUESTION question1 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "1");
            _surveyResult = new SURVEY_RESULT();

            try {
                // Add Question 1 choice
                _surveyResult.STD_QUESTION_CHOICE_ID = question1.STD_QUESTION_CHOICEs.FirstOrDefault().STD_QUESTION_CHOICE_ID;
                _surveyResult.RESULT_TEXT = txtDiagnosis.Text;
                survey.SURVEY_RESULTs.Add(_surveyResult);
            } catch (Exception surveyEx) {
                Exception _tbilogerror = HandleTBIException(surveyEx, "PatientGlobalImpOfChange.aspx", "100027", "SaveSurvey() - Question1");
            }
        }

        // Question 2 - Since beginning treatment at this clinic, how would you describe the change
        if (rblQ2.SelectedIndex > -1) {
            STD_QUESTION question2 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "2");
            _surveyResult = new SURVEY_RESULT();

            try {
                // Add Question 2 choice
                _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblQ2.SelectedValue);
                _surveyResult.RESULT_TEXT = rblQ2.SelectedItem.Text;
                survey.SURVEY_RESULTs.Add(_surveyResult);
            } catch (Exception surveyEx) {
                Exception _tbilogerror = HandleTBIException(surveyEx, "PatientGlobalImpOfChange.aspx", "100027", "SaveSurvey() - Question2");
            }
        }

        // Question 3 - In a similar way, please select the number below, that matches your degree of change since beginning care at this cinic 
        if (rblQ3.SelectedIndex > -1) {
            STD_QUESTION question3 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "3");
            _surveyResult = new SURVEY_RESULT();

            try {
                // Add Question 3 choice
                _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblQ3.SelectedValue);
                _surveyResult.RESULT_TEXT = rblQ3.SelectedItem.Text;
                survey.SURVEY_RESULTs.Add(_surveyResult);
            } catch (Exception surveyEx) {
                Exception _tbilogerror = HandleTBIException(surveyEx, "PatientGlobalImpOfChange.aspx", "100027", "SaveSurvey() - Question3");
            }
        }

        // Question RP
        _surveyResult = new SURVEY_RESULT();
        try
        {
            // Add Question DX choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(cntlRehabilitationProviderSelector.GetSelectedId());
            _surveyResult.RESULT_TEXT = cntlRehabilitationProviderSelector.GetSelectedResultText();
            survey.SURVEY_RESULTs.Add(_surveyResult);
        }
        catch (Exception surveyEx)
        {
            Exception _tbilogerror = HandleTBIException(surveyEx, "PatientGlobalImpOfChange.aspx", "100027", "SaveSurvey() - Question RP");
        }

        // Question TS
        _surveyResult = new SURVEY_RESULT();
        try
        {
            // Add Question DX choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(cntlTypeOfServiceSelector.GetSelectedId());
            _surveyResult.RESULT_TEXT = cntlTypeOfServiceSelector.GetSelectedResultText();
            survey.SURVEY_RESULTs.Add(_surveyResult);
        }
        catch (Exception surveyEx)
        {
            Exception _tbilogerror = HandleTBIException(surveyEx, "PatientGlobalImpOfChange.aspx", "100027", "SaveSurvey() - Question TS");
        }
    }// END SaveSurvey

    /// <summary>
    /// Initialize/setup controls
    /// </summary>
    /// <param name="survey"></param>
    private void SetUpSurvey(SURVEY survey) {
        try {
            foreach (var sResult in survey.SURVEY_RESULTs) {
                switch (sResult.STD_QUESTION_CHOICE.STD_QUESTION.QUESTION_NUMBER) {
                    case "DX":
                        if (sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1) {
                            ctrlDiagnosis.SelectedChoiceId = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();

                            if (sResult.STD_QUESTION_CHOICE.CHOICE_TEXT.ToLower().Contains("other")) {
                                ctrlDiagnosis.SetOtherDescriptionText(sResult.RESULT_TEXT);
                            }
                        } else {
                            ctrlDiagnosis.SelectedChoiceId = null;
                        }
                        break;
                    case "D":
                        if (!String.IsNullOrEmpty(sResult.RESULT_TEXT)) {
                            DateTime temp;
                            if (DateTime.TryParse(sResult.RESULT_TEXT, out temp)) {
                                dpDate1.SelectedDate = temp;
                            }
                        }
                        break;
                    case "1":
                        if (!String.IsNullOrEmpty(sResult.RESULT_TEXT)) {
                            txtDiagnosis.Text = sResult.RESULT_TEXT;
                        }
                        break;
                    case "2":
                        if (sResult.STD_QUESTION_CHOICE_ID != null) {
                            rblQ2.SelectedValue = sResult.STD_QUESTION_CHOICE_ID.ToString();
                        }
                        break;
                    case "3":
                        if (sResult.STD_QUESTION_CHOICE_ID != null) {
                            rblQ3.SelectedValue = sResult.STD_QUESTION_CHOICE_ID.ToString();
                        }
                        break;
                    case "RP":
                        if (sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1)
                        {
                            cntlRehabilitationProviderSelector.SelectedChoiceId = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        }
                        else
                        {
                            cntlRehabilitationProviderSelector.SelectedChoiceId = null;
                        }
                        break;
                   case "TS":
                        if (sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1)
                        {
                            cntlTypeOfServiceSelector.SelectedChoiceId = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();

                            if (sResult.STD_QUESTION_CHOICE.CHOICE_TEXT.ToLower().Contains("other"))
                            {
                                cntlTypeOfServiceSelector.SetOtherDescriptionText(sResult.RESULT_TEXT);
                            }
                        }
                        else
                        {
                            cntlTypeOfServiceSelector.SelectedChoiceId = null;
                        }
                        break;
                    default:
                        break;
                }
            }
        } catch (Exception ex) {
            Exception _tbilogerror = HandleTBIException(ex, "PatientGlobalImpOfChange.aspx", "100027", "SetUpSurvey()");
        }
    }// END SetUpSurvey

    /// <summary>
    /// Retrieve instrument title
    /// </summary>
    /// <param name="surveyID"></param>
    /// <returns></returns>
    private string GetInstrumentTitle(int surveyID) {
        string ret = String.Empty;

        try {
            List<string> instrumentTitle = _tbiInstrumentManager.GetInstrumentByID(surveyID).Title;
            if (instrumentTitle.Count > 0) {
                foreach (string titleLine in instrumentTitle) {
                    if (titleLine.Contains("PhD")) {
                        ret = ret + "<small>" + titleLine + "</small>" + "<br \\>";
                    } else {
                        ret = ret + titleLine + "<br \\>";
                    }
                }
            }

        } catch (Exception ex) {
            Exception _tbilogerror = HandleTBIException(ex, "PatientGlobalImpOfChange.aspx", "100027", "GetInstrumentTitle()");
            // Ignore error
        }

        return ret;
    }

    #endregion
}